"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductsService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const product_entity_1 = require("../../entities/product.entity");
const category_entity_1 = require("../../entities/category.entity");
const vendor_entity_1 = require("../../entities/vendor.entity");
let ProductsService = class ProductsService {
    productRepo;
    categoryRepo;
    vendorRepo;
    constructor(productRepo, categoryRepo, vendorRepo) {
        this.productRepo = productRepo;
        this.categoryRepo = categoryRepo;
        this.vendorRepo = vendorRepo;
    }
    toProductResponseDto(product) {
        if (!product) {
            throw new common_1.BadRequestException('Product cannot be null');
        }
        return {
            id: product.id,
            name: product.name,
            description: product.description,
            price: parseFloat(product.price?.toString() || '0'),
            image: product.image,
            stock: product.stock,
            isFeatured: product.isFeatured,
            categoryId: product.categoryId,
            vendorId: product.vendorId,
            createdAt: product.createdAt,
            updatedAt: product.updatedAt,
            category: product.category ? {
                id: product.category.id,
                name: product.category.name,
                description: product.category.description,
                image: product.category.image,
            } : undefined,
            vendor: product.vendor ? {
                id: product.vendor.id,
                name: product.vendor.name,
                email: product.vendor.email,
                isApproved: product.vendor.isApproved,
            } : undefined,
            variations: product.variations?.map(variation => ({
                id: variation.id,
                name: variation.name,
                price: parseFloat(variation.price?.toString() || '0'),
                stock: variation.stock,
                createdAt: variation.createdAt,
                updatedAt: variation.updatedAt,
            })) || [],
            reviews: product.reviews?.map(review => ({
                id: review.id,
                rating: review.rating,
                comment: review.comment,
                createdAt: review.createdAt,
            })) || [],
        };
    }
    async create(createProductDto, vendorId) {
        const existingProduct = await this.productRepo.findOne({
            where: { name: createProductDto.name },
        });
        if (existingProduct) {
            throw new common_1.ConflictException('Product with this name already exists');
        }
        if (createProductDto.categoryId) {
            const category = await this.categoryRepo.findOne({
                where: { id: createProductDto.categoryId },
            });
            if (!category) {
                throw new common_1.NotFoundException(`Category with ID ${createProductDto.categoryId} not found`);
            }
        }
        if (createProductDto.vendorId) {
            const vendor = await this.vendorRepo.findOne({
                where: { id: createProductDto.vendorId },
            });
            if (!vendor) {
                throw new common_1.NotFoundException(`Vendor with ID ${createProductDto.vendorId} not found`);
            }
            if (!vendor.isApproved) {
                throw new common_1.BadRequestException('Cannot create product for unapproved vendor');
            }
        }
        const finalVendorId = vendorId || createProductDto.vendorId;
        if (!finalVendorId) {
            throw new common_1.BadRequestException('Vendor ID is required');
        }
        const finalVendor = await this.vendorRepo.findOne({
            where: { id: finalVendorId },
        });
        if (!finalVendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${finalVendorId} not found`);
        }
        if (!finalVendor.isApproved) {
            throw new common_1.BadRequestException('Cannot create product for unapproved vendor');
        }
        const product = this.productRepo.create({
            ...createProductDto,
            vendorId: finalVendorId,
        });
        const savedProduct = await this.productRepo.save(product);
        const completeProduct = await this.productRepo.findOne({
            where: { id: savedProduct.id },
            relations: ['category', 'vendor', 'variations', 'reviews'],
        });
        if (!completeProduct) {
            throw new common_1.NotFoundException(`Product with ID ${savedProduct.id} not found`);
        }
        return this.toProductResponseDto(completeProduct);
    }
    async findAll(query) {
        const { page = 1, limit = 10, categoryId, vendorId, minPrice, maxPrice, search, isFeatured, inStock, sortBy = 'createdAt', sortOrder = 'DESC', } = query;
        const skip = (page - 1) * limit;
        const where = {};
        if (categoryId) {
            where.categoryId = categoryId;
        }
        if (vendorId) {
            where.vendorId = vendorId;
        }
        if (minPrice !== undefined || maxPrice !== undefined) {
            where.price = (0, typeorm_2.Between)(minPrice !== undefined ? minPrice : 0, maxPrice !== undefined ? maxPrice : 999999);
        }
        if (isFeatured !== undefined) {
            where.isFeatured = isFeatured;
        }
        if (inStock !== undefined) {
            if (inStock) {
                where.stock = (0, typeorm_2.MoreThan)(0);
            }
            else {
                where.stock = 0;
            }
        }
        if (search) {
            where.name = (0, typeorm_2.Like)(`%${search}%`);
        }
        const order = {};
        order[sortBy] = sortOrder.toUpperCase();
        const [products, total] = await this.productRepo.findAndCount({
            where,
            relations: ['category', 'vendor', 'variations', 'reviews'],
            order,
            skip,
            take: limit,
        });
        const totalPages = Math.ceil(total / limit);
        return {
            products: products.map(product => this.toProductResponseDto(product)),
            total,
            page,
            limit,
            totalPages,
        };
    }
    async findOne(id) {
        const product = await this.productRepo.findOne({
            where: { id },
            relations: ['category', 'vendor', 'variations', 'reviews'],
        });
        if (!product) {
            throw new common_1.NotFoundException(`Product with ID ${id} not found`);
        }
        return this.toProductResponseDto(product);
    }
    async findByIds(ids) {
        if (!ids || ids.length === 0) {
            return [];
        }
        const products = await this.productRepo.find({
            where: { id: (0, typeorm_2.In)(ids) },
            relations: ['category', 'vendor', 'variations', 'reviews'],
        });
        return products.map(product => this.toProductResponseDto(product));
    }
    async update(id, updateProductDto, vendorId) {
        const product = await this.productRepo.findOne({
            where: { id },
            relations: ['category', 'vendor', 'variations', 'reviews'],
        });
        if (!product) {
            throw new common_1.NotFoundException(`Product with ID ${id} not found`);
        }
        if (vendorId && product.vendorId !== vendorId) {
            throw new common_1.BadRequestException('You can only update your own products');
        }
        if (updateProductDto.name && updateProductDto.name !== product.name) {
            const existingProduct = await this.productRepo.findOne({
                where: { name: updateProductDto.name },
            });
            if (existingProduct && existingProduct.id !== id) {
                throw new common_1.ConflictException('Product with this name already exists');
            }
        }
        if (updateProductDto.categoryId) {
            const category = await this.categoryRepo.findOne({
                where: { id: updateProductDto.categoryId },
            });
            if (!category) {
                throw new common_1.NotFoundException(`Category with ID ${updateProductDto.categoryId} not found`);
            }
        }
        if (updateProductDto.vendorId) {
            const vendor = await this.vendorRepo.findOne({
                where: { id: updateProductDto.vendorId },
            });
            if (!vendor) {
                throw new common_1.NotFoundException(`Vendor with ID ${updateProductDto.vendorId} not found`);
            }
            if (!vendor.isApproved) {
                throw new common_1.BadRequestException('Cannot assign product to unapproved vendor');
            }
        }
        Object.assign(product, updateProductDto);
        const updatedProduct = await this.productRepo.save(product);
        const completeProduct = await this.productRepo.findOne({
            where: { id: updatedProduct.id },
            relations: ['category', 'vendor', 'variations', 'reviews'],
        });
        if (!completeProduct) {
            throw new common_1.NotFoundException(`Product with ID ${updatedProduct.id} not found`);
        }
        return this.toProductResponseDto(completeProduct);
    }
    async remove(id, vendorId) {
        const product = await this.productRepo.findOne({
            where: { id },
            relations: ['orderDetails', 'category', 'vendor', 'variations', 'reviews'],
        });
        if (!product) {
            throw new common_1.NotFoundException(`Product with ID ${id} not found`);
        }
        if (vendorId && product.vendorId !== vendorId) {
            throw new common_1.BadRequestException('You can only delete your own products');
        }
        if (Array.isArray(product.orderDetails) && product.orderDetails.length > 0) {
            throw new common_1.ConflictException('Cannot delete product with order history. Deactivate it instead.');
        }
        await this.productRepo.remove(product);
    }
    async updateStock(id, quantity, operation = 'decrease') {
        if (quantity <= 0) {
            throw new common_1.BadRequestException('Quantity must be greater than 0');
        }
        const product = await this.productRepo.findOne({ where: { id } });
        if (!product) {
            throw new common_1.NotFoundException(`Product with ID ${id} not found`);
        }
        if (operation === 'decrease') {
            if (product.stock < quantity) {
                throw new common_1.BadRequestException(`Insufficient stock. Available: ${product.stock}, Requested: ${quantity}`);
            }
            product.stock -= quantity;
        }
        else {
            product.stock += quantity;
        }
        const updatedProduct = await this.productRepo.save(product);
        const completeProduct = await this.productRepo.findOne({
            where: { id: updatedProduct.id },
            relations: ['category', 'vendor', 'variations', 'reviews'],
        });
        if (!completeProduct) {
            throw new common_1.NotFoundException(`Product with ID ${updatedProduct.id} not found`);
        }
        return this.toProductResponseDto(completeProduct);
    }
    async toggleFeatured(id) {
        const product = await this.productRepo.findOne({ where: { id } });
        if (!product) {
            throw new common_1.NotFoundException(`Product with ID ${id} not found`);
        }
        product.isFeatured = !product.isFeatured;
        const updatedProduct = await this.productRepo.save(product);
        const completeProduct = await this.productRepo.findOne({
            where: { id: updatedProduct.id },
            relations: ['category', 'vendor', 'variations', 'reviews'],
        });
        if (!completeProduct) {
            throw new common_1.NotFoundException(`Product with ID ${updatedProduct.id} not found`);
        }
        return this.toProductResponseDto(completeProduct);
    }
    async getFeaturedProducts(limit = 10) {
        const products = await this.productRepo.find({
            where: { isFeatured: true, stock: (0, typeorm_2.MoreThan)(0) },
            relations: ['category', 'vendor', 'variations', 'reviews'],
            order: { createdAt: 'DESC' },
            take: limit,
        });
        return products.map(product => this.toProductResponseDto(product));
    }
    async getProductsByCategory(categoryId, query) {
        const category = await this.categoryRepo.findOne({ where: { id: categoryId } });
        if (!category) {
            throw new common_1.NotFoundException(`Category with ID ${categoryId} not found`);
        }
        return this.findAll({
            ...query,
            categoryId,
        });
    }
    async getProductsByVendor(vendorId, query) {
        const vendor = await this.vendorRepo.findOne({ where: { id: vendorId } });
        if (!vendor) {
            throw new common_1.NotFoundException(`Vendor with ID ${vendorId} not found`);
        }
        return this.findAll({
            ...query,
            vendorId,
        });
    }
    async searchProducts(searchTerm, limit = 20) {
        if (!searchTerm || searchTerm.trim().length === 0) {
            throw new common_1.BadRequestException('Search term is required');
        }
        const products = await this.productRepo.find({
            where: [
                { name: (0, typeorm_2.Like)(`%${searchTerm}%`) },
                { description: (0, typeorm_2.Like)(`%${searchTerm}%`) },
            ],
            relations: ['category', 'vendor', 'variations', 'reviews'],
            order: { name: 'ASC' },
            take: limit,
        });
        return products.map(product => this.toProductResponseDto(product));
    }
    async getLowStockProducts(threshold = 10) {
        if (threshold < 0) {
            throw new common_1.BadRequestException('Threshold must be a non-negative number');
        }
        const products = await this.productRepo.find({
            where: { stock: (0, typeorm_2.LessThanOrEqual)(threshold) },
            relations: ['category', 'vendor', 'variations', 'reviews'],
            order: { stock: 'ASC' },
        });
        return products.map(product => this.toProductResponseDto(product));
    }
    async getProductStats() {
        const totalProducts = await this.productRepo.count();
        const totalInStock = await this.productRepo.count({ where: { stock: (0, typeorm_2.MoreThan)(0) } });
        const totalOutOfStock = await this.productRepo.count({ where: { stock: 0 } });
        const featuredProducts = await this.productRepo.count({ where: { isFeatured: true } });
        const categoryStats = await this.productRepo
            .createQueryBuilder('product')
            .select('category.name', 'categoryName')
            .addSelect('COUNT(*)', 'count')
            .leftJoin('product.category', 'category')
            .groupBy('category.name')
            .getRawMany();
        const vendorStats = await this.productRepo
            .createQueryBuilder('product')
            .select('vendor.name', 'vendorName')
            .addSelect('COUNT(*)', 'count')
            .leftJoin('product.vendor', 'vendor')
            .groupBy('vendor.name')
            .getRawMany();
        return {
            totalProducts,
            totalInStock,
            totalOutOfStock,
            featuredProducts,
            categoryStats,
            vendorStats,
        };
    }
};
exports.ProductsService = ProductsService;
exports.ProductsService = ProductsService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(product_entity_1.Product)),
    __param(1, (0, typeorm_1.InjectRepository)(category_entity_1.Category)),
    __param(2, (0, typeorm_1.InjectRepository)(vendor_entity_1.Vendor)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository])
], ProductsService);
//# sourceMappingURL=products.service.js.map